import React from 'react';
import { View, Text, StyleSheet, Alert, BackHandler, StatusBar, Linking } from 'react-native';
import { Image } from 'react-native-elements';
import Axios from 'axios';
import { TouchableOpacity, ScrollView } from 'react-native-gesture-handler';
import HTML from 'react-native-render-html';
import { connect } from 'react-redux';
import i18n from 'i18n-js';
import { FontAwesome, AntDesign } from '@expo/vector-icons';
import Toast from 'react-native-tiny-toast';
import requestParams from '../helper/requestParams';
import CustomIosStatusBar from './CustomIosStatusBar';

class InboxDetail extends React.Component {
    constructor(props) {
        super(props);
        this.handleBackButtonClick = this.handleBackButtonClick.bind(this)
        this.state = {
            title: '',
            subtitle: '',
            image: '',
            content: '',
            image_height: 400,
            statusbar_height: 50
        }
    }

    componentDidMount() {
        BackHandler.addEventListener('hardwareBackPress', this.handleBackButtonClick);
        this.getInboxDetail()
    }

    componentWillUnmount() {
        BackHandler.removeEventListener('hardwareBackPress', this.handleBackButtonClick);
    }

    handleBackButtonClick() {
        this.props.navigation.navigate('Inbox List');
        return true;
    }

    getInboxDetail() {

        let params = Object.assign(requestParams, {
            session_id: this.props.session_id,
            message_id: this.props.route.params.idInbox
        })

        // console.log(params)

        Axios.post(this.props.BASE_URL + 'crm/v2/inbox/get_detail', params).then(res => {
            let data = res.data.message
            console.log(data)

            this.setState({
                title: data.message.title,
                subtitle: data.message.title,
                image: data.message.content_image,
                content: data.message.content,
            })
            // console.log(this.state.content)

        }).catch(error => {
            let response = error.response.data;
            Alert.alert("Error", response.msg);
            this.setState({
                indicator: false,
            })

            // Alert.alert(response.msg);
        })
    }

    handleDeleteInbox() {
        Alert.alert(
            "",
            i18n.t('alertDeleteInbox'),
            [
                {
                    text: i18n.t('alertno'),
                    onPress: () => console.log('Cancel Renewal'),
                    style: 'cancel',
                },
                {
                    text: i18n.t('alertyes'), onPress: () => this.deleteInbox()
                },
            ],
            { cancelable: false },
        )
    }

    deleteInbox() {
      
        let params = Object.assign(requestParams, {
            session_id: this.props.session_id,
            message_id: this.props.route.params.idInbox
        })
        // console.log(params)

        Axios.post(this.props.BASE_URL + 'crm/v2/inbox/delete', params).then(res => {
            let data = res.data
            console.log(data)
            Toast.show(i18n.t('alertSuccessDeleteInbox'))
            this.props.navigation.navigate('Inbox List')

        }).catch(error => {
            let response = error.response.data;
            Alert.alert("Error", response.msg);
            this.setState({
                indicator: false,
            })

            // Alert.alert(response.msg);
        })
    }

    render() {
        return (
            <View style={styles.container}>
                {Platform.OS === 'ios' ? (
                <CustomIosStatusBar color='#CFB368' />
                ) : (
                    <StatusBar backgroundColor='#CFB368' />
                )}
                <View style={styles.header}>
                    <View style={{ padding: 15, flex: 0.1, alignItems: 'flex-start' }}>
                        <TouchableOpacity onPress={() => this.props.navigation.navigate('Inbox List')}>
                            <AntDesign name="arrowleft" size={22} color="white" />
                        </TouchableOpacity>
                    </View>
                    <View style={{ flex: 0.8, alignItems: 'center', padding:15 }}>
                        <Text style={{ fontSize: 20, fontFamily: 'Gotham-Black', color: 'white' }}>{i18n.t('headerInboxDetail')}</Text>
                    </View>
                    <View style={{ padding: 15, flex: 0.1, alignItems: 'flex-end' }}>
                        <TouchableOpacity onPress={() => this.handleDeleteInbox()}>
                            <FontAwesome name="trash-o" size={24} color="white" />
                        </TouchableOpacity>
                    </View>
                </View>
                <ScrollView style={styles.body}>
                    <View style={{ flex: 1 }}>
                        <Image style={{ height: 360, width: '100%' }} resizeMode={'contain'} source={this.state.image ? { uri: this.state.image } : null} />
                    </View>
                    <View style={styles.content}>
                        <Text style={styles.title}>{this.state.title}</Text>
                        <View style={{ backgroundColor: '#adadad', height: 1.5, flex: 1, marginBottom: 5 }} />
                        <HTML html={this.state.content} tagsStyles={{
                            p: { fontSize: 12, color: '#838383', fontFamily: 'Gotham-Light', margin: 1.5 },
                            blockquote: { backgroundColor: "#f1f1f1", padding: 12, paddingBottom: 0, marginTop: 0 },
                            li: { fontSize: 12, color: '#838383', fontFamily: 'Gotham-Light', marginBottom: 0 },
                            ul: { fontSize: 12, color: '#838383', fontFamily: 'Gotham-Light', marginVertical: 5, marginBottom: -10 },

                        }}
                            ignoredTags={['br']}
                            ignoredStyles={['font-family', '&nbsp;', '&amp;']}
                            onLinkPress={(event, href) => {
                                Linking.openURL(href)
                            }} />
                    </View>
                </ScrollView>

            </View>
        )
    }
}

const styles = StyleSheet.create({
    container: {
        flex: 1,
        backgroundColor: 'white',
    },
    header: {
        flex: 0.1,
        flexDirection: 'row',
        backgroundColor: '#CFB368',
        alignItems: 'center',
        paddingTop: StatusBar.currentHeight
    },
    headerTitle: {
        flex: 1,
        textAlign: 'center',
        margin: 20,
        fontSize: 23,
        alignSelf: 'center',
        color: 'white',
        fontFamily: 'Gotham-Black'
    },
    image_container: (height) => ({
        height: height,
        backgroundColor: 'gray'
    }),
    image: (height) => ({
        width: '100%',
        height: height
    }),
    title: {
        textAlign: 'center',
        fontFamily: 'Gotham-Black',
        fontSize: 22,
        marginBottom: 10,
        color: '#CFB368'
    },
    content: {
        paddingHorizontal: 25,
        paddingVertical: 30
    },
    body: {
        flex: 0.9
    }

})

const mapStateToProps = (state) => {
    return {

        session_id: state.session_id,
        language: state.language,
        BASE_URL: state.BASE_URL,

    }
}

export default connect(mapStateToProps)(InboxDetail);